/*************************************************************************
 *
 * \file: util_mutex_robust.h
 *
 * \release: $Name:$
 *
 * <brief description>.
 * Declares the robust variant of the pthread_mutex in a simplified way
 * <detailed description>
 * \component: utility - util_mutex_robust
 *
 * \author: Dhanasekaran D / RBEI/ECF3 / dhanasekaran.d(o)in.bosch.com
 *
 * \copyright (c) 2015 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 * \see <related items>
 *
 * \history
 *
 ***********************************************************************/

#ifndef __UTIL_MUTEX_ROBUST__
#define __UTIL_MUTEX_ROBUST__

#include <pthread.h>
#include <sys/types.h>

#ifdef __cplusplus
extern "C" {
#endif

#define UTIL_MUTEX_RECOVERED              0
#define UTIL_MUTEX_RECOVERED_NO_HANDLER   1
#define UTIL_MUTEX_RECOVERY_FAILED      (-1)

typedef struct __util_mutex_robust util_mutex_robust;

/**
 * State consistent handler function. Based on the previous
 * critical section where the previous OWNER has terminated
 * the recovery of the shared resources should be done here.
 *
 * \param mutex     robust mutex object
 *
 * \returns         0 - successfully recovered, mutex can continue to be used
 *					non-zero - failed to recover, mutex can't be used
 */
typedef int (*mutex_robust_recovery_handler)(util_mutex_robust *);


/**
 * robust mutex object
 */
struct __util_mutex_robust {
	pthread_mutex_t m;
	/*last incomplete critical section*/
	int section;
	/*callback for recovering from EOWNERDEAD*/
	mutex_robust_recovery_handler recovery_handler;
	/*private data specific to mutex*/
	void *pdata;
};


/**
 * Initialize pthread mutex with robust and process_shared attribute.
 *
 * \param mutex     robust mutex object
 * \param handler   State consistent handler function. Based on the previous
 *                  critical section where the previous OWNER has terminated
 *                  the recovery of the shared resources should be done here.
 *
 * \returns         pthread_mutex_init return value
 *
 * \remarks
 */
int UTIL_mutex_robust_init(util_mutex_robust *mutex,
							mutex_robust_recovery_handler handler);


/**
 * Acquires the mutex lock
 *
 * After the pthread_mutex_lock function returns, it checks the error code and
 * handles the EOWNERDEAD by calling the state recovery handler function
 * registered during mutex init process. If successfully recovered then
 * function returns 0, otherwise pthread_mutex_unlock is called and non-zero
 * value is returned. In unsuccessful recovery case, the mutex will not be in
 * usable state.
 *
 * \param mutex     robust mutex object
 *
 * \returns         UTIL_MUTEX_RECOVERY_SUCCESS - successfully recovered by
 *                        the recovery handler function.
 *                   UTIL_MUTEX_RECOVERED_NO_HANDLER - pthread_mutex_consistent
 *                       called without a recovered handler.
 *                   UTIL_MUTEX_RECOVERY_FAILED - Recovery failed in the handler
 *                       function.
 *                   Others values - same as pthread_mutex_lock() return
 *
 * \remarks         Writing an efficient state consistent handler function is
 *                  important, because otherwise system likely to be reset to
 *                  recover from the situation.
 */
int UTIL_mutex_robust_lock(util_mutex_robust *mutex);


/**
 * Releases the mutex lock
 *
 * \param mutex     robust mutex object
 *
 * \returns         same as pthread_mutex_unlock
 *
 * \remarks
 */
int UTIL_mutex_robust_unlock(util_mutex_robust *mutex);


/**
 * Destroys the pthread mutex
 *
 * \param mutex     robust mutex object
 *
 * \returns         same as pthread_mutex_destroy
 *
 * \remarks
 */
int UTIL_mutex_robust_destroy(util_mutex_robust *mutex);

#ifdef __cplusplus
};
#endif

#endif /*__UTIL_MUTEX_ROBUST__*/
